#!/bin/sh

get_hypervisor() {
    $WG_BINDIR/python - "$@" <<EOF
import json, sys
try:
    input = open('/run/watchguard/vm-host.json', 'r')
    dict = json.load(input)
except:
    dict = {}
hypervisor = dict.get('vm-vendor', '$WG_VM_VENDOR')
print(hypervisor)
EOF
}

get_ipv4_netaddr() {
    $WG_BINDIR/python - "$@" <<EOF
import sys
from wg_system import utils
address = sys.argv[1]
masklen = sys.argv[2]
network = utils.get_ipv4_netaddr(address, int(masklen))
print(network)
EOF
}

get_ipv4_host_is_all_ones() {
    $WG_BINDIR/python - "$@" <<EOF
import sys
from wg_system import utils
string = sys.argv[1]
masklen = sys.argv[2]
result = utils.get_ipv4_host_number_is_all_ones(string, int(masklen))
print(result)
EOF
}

get_ipv4_host_is_all_zeros() {
    $WG_BINDIR/python - "$@" <<EOF
import sys
from wg_system import utils
string = sys.argv[1]
masklen = sys.argv[2]
result = utils.get_ipv4_host_number_is_all_zeros(string, int(masklen))
print(result)
EOF
}

parse_arguments() {
    ARGS=`getopt -o "cdg:hi:m:" -l "check,dhcp,gateway:,help,ip:,mask:" -n "$SCRIPT" -- "$@"`

    if [ $? -ne 0 ]; then
        print_usage_and_quit
    fi

    eval set -- "$ARGS"

    while true; do
        case "$1" in

            -c|--check)
                IFCHECK=true
                shift
                ;;
            
            -d|--dhcp)
                if [ "$IFMETHOD" = static ]; then
                    print_usage_and_quit
                fi
                IFMETHOD=dhcp
                shift
                ;;
            
            -g|--gateway)
                if [ "$IFMETHOD" = dhcp ]; then
                    print_usage_and_quit
                fi
                IFMETHOD=static
                IFGATEWAY=$2
                shift 2
                ;;
            
            -h|--help)
                print_usage
                shift
                exit 0
                ;;
            
            -i|--ip)
                if [ "$IFMETHOD" = dhcp ]; then
                    print_usage_and_quit
                fi
                IFMETHOD=static
	        IFNETADDR=$2
                shift 2
                ;;
            
            -m|--mask)
                if [ "$IFMETHOD" = dhcp ]; then
                    print_usage_and_quit
                fi
                IFMETHOD=static
	        IFNETMASK=$2
                shift 2
                ;;
            
            --)
                shift
                break
                ;;

        esac
    done

    if [ "$#" -gt 0 ]; then
        IFNAME=$1
    fi
}

print_configuration() {
    case "$IFMETHOD" in

	dhcp) print_configuration_dhcp
            ;;

	static) print_configuration_static_ip
            ;;

    esac
}

print_configuration_dhcp() {
    cat <<EOF
Method: DHCP
EOF
}

print_configuration_static_ip() {
    cat <<EOF
Method: Static IP
Static IP: $IFNETADDR
Mask: $IFNETMASK
Gateway: $IFGATEWAY
EOF
}

print_error_and_quit() {
    echo "ERROR: $*" >&2
    exit 1
}

print_usage() {
    cat <<EOF >&2
usage: $SCRIPT -d [<NAME>]
       $SCRIPT --dhcp [<NAME>]
       $SCRIPT -i <ADDR> -m <MASK> -g <GATEWAY> [<NAME>]
       $SCRIPT --ip <ADDR> --mask <MASK> --gateway <GATEWAY> [<NAME>]

where: <NAME> is a network interface name (e.g., eth0 is default)
       <ADDR> is an IPv4 address in dotted-quad notation (e.g., 192.168.1.101)
       <MASK> is a network mask or routing prefix length in bits (e.g., 24)
       <GATEWAY> is an IPv4 address in dotted-quad notation (e.g., 192.168.1.1)
EOF
}

print_usage_and_quit() {
    print_usage
    exit 1
}

save_and_clean_up() {
    if [ "$IFCHECK" = true ]; then
        cat $TMPFILE
    else
        /bin/cp -f "${TMPFILE}" "$WG_ETCDIR/system/interface-${IFNAME}.json"
    fi
}

set_configuration() {
    case "$IFMETHOD" in

	dhcp) set_configuration_dhcp
            ;;

	static) set_configuration_static_ip
            ;;

    esac
}

set_configuration_dhcp() {
    cat <<EOF >$TMPFILE
{
    "enabled": true,
    "inet": {
        "method": "${IFMETHOD}"
    }
}
EOF
}

set_configuration_static_ip() {
    cat <<EOF >$TMPFILE
{
    "enabled": true,
    "inet": {
        "method": "${IFMETHOD}",
        "static-ip": {
            "ip": "${IFNETADDR}",
            "mask": ${IFNETMASK},
            "gateway": "${IFGATEWAY}"
        }
    }
}
EOF
}

validate_arguments() {
    if [ -z "$IFMETHOD" ]; then
        print_usage_and_quit
    fi

    if [ "$IFMETHOD" = static ]; then

        if [ -z "$IFNETADDR" -o -z "$IFNETMASK" -o -z "$IFGATEWAY"  ]; then
            print_usage_and_quit
        elif ! echo $IFNETMASK | egrep '^[1-9][0-9]{0,1}$' >/dev/null; then
            print_error_and_quit "$IFNETMASK: Invalid network mask length"
        elif [ "$IFNETMASK" -lt 1 -o "$IFNETMASK" -gt 31 ]; then
            print_error_and_quit "$IFNETMASK: Invalid network mask length"
        elif [ "$IFNETADDR" = "$IFGATEWAY" ]; then
            print_error_and_quit "$IFNETADDR: Address same as gateway"
        fi

        for addr in $IFNETADDR $IFGATEWAY; do
            
            if ! echo $addr | egrep '^((0|[1-9][0-9]{0,2})\.){3}(0|[1-9][0-9]{0,2})$' >/dev/null; then
                print_error_and_quit "$addr: Invalid IP address"
            fi

            index=0


            for octet in `echo $addr | sed 's/\./ /g'`; do
                
                if [ "$octet" -gt 255 ]; then
                    print_error_and_quit "$addr: Invalid IP address"
                fi
                
                eval octet$index=$octet
                index=$((index + 1))
            done

            # following tests are derived from Reserved IPv4 addresses table in Wikipedia
            # (http://en.wikipedia.org/wiki/Reserved_IP_addresses#Reserved_IPv4_addresses)
            
            if [ $octet0 -eq 255 -a $octet1 -eq 255 -a $octet2 -eq 255 -a $octet3 -eq 255 ]; then
                print_error_and_quit "$addr: Reserved for limited broadcast (RFC 5735)"
            elif [ $octet0 -eq 0 ]; then
                print_error_and_quit "$addr: Used for broadcast only (RFC 1700)"
            elif [ $octet0 -eq 127 ]; then
                print_error_and_quit "$addr: Used for loopback only (RFC 5735)"
            elif [ $octet0 -eq 169 -a $octet1 -eq 254 ]; then
                print_error_and_quit "$addr: Used for autoconfiguration only (RFC 5735)"
            elif [ $octet0 -ge 224 -a $octet0 -le 239 ]; then
                print_error_and_quit "$addr: Reserved for multicast (RFC 5735)"
            elif [ "`get_ipv4_host_is_all_ones $addr $IFNETMASK`" = True ]; then
                print_error_and_quit "$addr/$IFNETMASK: Host number reserved for broadcast"
            elif [ "`get_ipv4_host_is_all_zeros $addr $IFNETMASK`" = True ]; then
                print_error_and_quit "$addr/$IFNETMASK: Host number reserved for network"
            fi
            
        done

        GWNETWORK=`get_ipv4_netaddr $IFGATEWAY $IFNETMASK`
        IPNETWORK=`get_ipv4_netaddr $IFNETADDR $IFNETMASK`

        if [ "$GWNETWORK" != "$IPNETWORK" ]; then
            print_error_and_quit "$IFGATEWAY: Gateway not in network $IFNETADDR/$IFNETMASK"
        fi

    fi

    case "$IFNAME" in
        
        eth*|lan*)
            if ! fgrep "\"$IFNAME\"" $IFINDEX >/dev/null; then
                print_error_and_quit "$IFNAME: No such interface detected"
            fi
            ;;
        
        *)
            print_error_and_quit "$IFNAME: Invalid interface name"
            ;;
        
    esac
}

#########################
# Execution begins here #
#########################

set -eu
umask 002

SCRIPT="`basename $0`"

SYSTEM_DEFAULTS=/etc/default/wg_system
VMHOST_DEFAULTS=/etc/default/wg_vm_host

test -r $SYSTEM_DEFAULTS && . $SYSTEM_DEFAULTS
test -r $VMHOST_DEFAULTS && . $VMHOST_DEFAULTS

HYPERVISOR=`get_hypervisor`
IFCHECK=false
IFINDEX="${WG_ETCDIR}/system/interfaces.json"
IFNAME=eth0
IFMETHOD=
IFNETADDR=
IFNETMASK=
IFGATEWAY=

TMPFILE="/tmp/interface-$$.json"
trap "test -f $TMPFILE && /bin/rm -f $TMPFILE 2>/dev/null" EXIT

case "$HYPERVISOR" in
    (Amazon-EC2|Microsoft-Azure)
        print_error_and_quit "Operation not permitted on $HYPERVISOR"
        ;;
    (*)
        ;;
esac

if [ ! -r $IFINDEX ]; then
    print_error_and_quit "No interfaces detected"
fi

parse_arguments "$@"
validate_arguments
print_configuration
set_configuration
save_and_clean_up

exit 0
