#!/bin/sh

set -e

update_apt_keyring() {
    if [ -f "/root/trustedkeys.gpg" ] ; then 
        gpg --homedir /root --no-default-keyring --keyring /root/trustedkeys.gpg --export cm@watchguard.com | apt-key add -
    fi
}

migrate_smtp_config() {
    WG_PYTHON_BIN=${WG_BINDIR}/python
    WG_CONFIG_PYC=${WG_BINDIR}/wg_system_config.pyc
    WG_POSTFIX_PYC=${WG_LIBDIR}/python2.7/site-packages/wg_system/postfix.pyc
    WG_POSTFIX_CFG=${WG_ETCDIR}/system/postfix.json
    SMTP_INI_FILE=${WG_ETCDIR}/wlogserver/wlogserver.ini

    if [ -r ${WG_POSTFIX_PYC} -a -r ${SMTP_INI_FILE} ] ; then
        echo "Migrating SMTP client configuration"
        ${WG_PYTHON_BIN} ${WG_POSTFIX_PYC} -i ${SMTP_INI_FILE} migrateconfig
        echo "Applying SMTP client configuration"
        ${WG_PYTHON_BIN} ${WG_CONFIG_PYC} ${WG_POSTFIX_CFG}
    fi
}

reconfig_postfix() {
    WG_PYTHON_BIN=${WG_BINDIR}/python
    WG_CONFIG_PYC=${WG_BINDIR}/wg_system_config.pyc
    WG_POSTFIX_CFG=${WG_ETCDIR}/system/postfix.json
    POSTFIX_MAIN_CF="/etc/postfix/main.cf"
    
    if [ -r ${POSTFIX_MAIN_CF} ] ; then
        if ! grep -q '^inet_protocols' ${POSTFIX_MAIN_CF} ; then
            echo "Restart postfix"
            ${WG_PYTHON_BIN} ${WG_CONFIG_PYC} ${WG_POSTFIX_CFG}
        fi
    fi
}

remove_weak_ssh_moduli() {
    srcf="/etc/ssh/moduli"
    tmpf=`mktemp`
    /usr/bin/awk '$5 >= 2047' $srcf > $tmpf
    if /usr/bin/cmp -s $srcf $tmpf ; then
        /bin/rm -f $tmpf
    else
        /bin/cp -fp $srcf ${srcf}.bak
        /bin/mv -f $tmpf $srcf
        chmod 644 $srcf
        chown --reference=/etc/ssh $srcf
    fi
}

check_sshd_config() {
    SSHD_CONFIG="/etc/ssh/sshd_config"
    SSHD_JSON="${WG_ETCDIR}/system/sshd.json"
    if ! grep -q '^Ciphers' ${SSHD_CONFIG} ; then
        # Ciphers directive is missing from /etc/ssh/sshd_config. We need
        # to regenerate it, if support SSH access is enabled. 
        if [ -f ${SSHD_JSON} ] ; then
            touch ${SSHD_JSON}
        fi
    fi
}

usage() {
    echo "usage: $1 install|upgrade|uninstall" >&2
    exit 1
}

if [ $# -ne 1 ] ; then
    usage $0
fi

case "$1" in
    "install" | "upgrade")

        if [ -r /etc/default/wg_system ] ; then
            . /etc/default/wg_system
        else
            # Nothing to do
            exit 0
        fi

        if [ ! -x /sbin/initctl ] && [ ! -x /usr/bin/systemctl ] ; then
            # Upstart and Systemd not installed. Nothing to do at the moment.
            exit 0
        fi

        # Check if wg_systemd is already registered and make sure it is not
        # running
        if [ -f /etc/init/wg_systemd.conf ] ; then
            echo "Stopping wg_systemd"
            /sbin/initctl stop wg_systemd >/dev/null || true
        fi
        if [ -f ${WG_SYSTEMDDIR}/wg-systemd.service ] ; then
            echo "Stopping wg_systemd"
            /usr/bin/systemctl stop wg-systemd >/dev/null || true
        fi

        # Do the same for wg_backup
        if [ -f /etc/init/wg_backup.conf ] ; then
            echo "Stopping wg_backup"
            /sbin/initctl stop wg_backup >/dev/null || true
        fi
        if [ -f ${WG_SYSTEMDDIR}/wg-backup.service ] ; then
            echo "Stopping wg_backup"
            /usr/bin/systemctl stop wg-backup >/dev/null || true
        fi

        # Create $WG_ADMIN_GROUP system group
        rc=0
        getent group "$WG_ADMIN_GROUP" >/dev/null || rc=$?
        if [ $rc -eq 2 ] ; then
            echo "Creating $WG_ADMIN_GROUP system group"
            groupadd -r "$WG_ADMIN_GROUP"
        fi

        # Create or modify $WG_SUPPORT_USER system account
        rc=0
        getent passwd "$WG_SUPPORT_USER" >/dev/null || rc=$?
        if [ $rc -eq 2 ] ; then

            # Create $WG_SUPPORT_USER system account with password
            # "readwrite"
            echo "Creating $WG_SUPPORT_USER system account"
            password='$6$xzv66s/J$gIYEf07igw9lAruhGDZq4leDBwe3sy28SAu17KJ78sZoTqJRAEhK/bsHeQiQEa0SZuBWEC9zVXp97rmRGG/cI1'
            useradd -mr -g "$WG_ADMIN_GROUP" -p "$password" -s /bin/bash "$WG_SUPPORT_USER"

            # Force $WG_SUPPORT_USER to set new password at first login
            chage -d 0 "$WG_SUPPORT_USER"

        elif [ $rc -eq 0 ] ; then

            # Make sure $WG_SUPPORT_USER system account has
            # $WG_ADMIN_GROUP as the primary group.
            echo "Setting $WG_SUPPORT_USER system account primary group to $WG_ADMIN_GROUP system group"
            usermod -g "$WG_ADMIN_GROUP" "$WG_SUPPORT_USER" 2>&1 | grep -v "usermod: no changes" || true

        fi

        # Check group ownership and set-group-id for /usr/local/share/ca-certificates
        if [ -d /usr/local/share/ca-certificates ] ; then
            if [ "`ls -ld /usr/local/share/ca-certificates | awk '{print $4}'`" != "$WG_ADMIN_GROUP" ] ; then
                echo "Fixing directory /usr/local/share/ca-certificates ownership/permissions"
                dpkg-statoverride --force --update --add root $WG_ADMIN_GROUP 2775 /usr/local/share/ca-certificates
            fi
        fi

        # Check for existence of group adm
        if getent group adm >/dev/null ; then

            # Add $WG_SUPPORT_USER system account to adm group
            echo "Adding $WG_SUPPORT_USER system account to adm system group"
            usermod -a -G adm "$WG_SUPPORT_USER"

            # Work-around (hack) for bug in upstart
            # (https://bugs.launchpad.net/ubuntu/+source/upstart/+bug/1074564)

            # Check for directory /var/log/upstart with gid adm
            if [ -d /var/log/upstart ] ; then
                if [ "`ls -ld /var/log/upstart | awk '{print $4}'`" != "adm" ] ; then
                    echo "Fixing directory /var/log/upstart ownership/permissions"
                    chgrp -R adm /var/log/upstart
                    chmod g+s /var/log/upstart
                fi
            fi

        fi

        # Check for existence of group fuse
        if getent group fuse >/dev/null ; then
            echo "Adding $WG_ADMIN_USER system account to fuse system group"
            usermod -a -G fuse "$WG_ADMIN_USER" 2>&1 | grep -v "usermod: no changes" || true
            echo "Adding $WG_SUPPORT_USER system account to fuse system group"
            usermod -a -G fuse "$WG_SUPPORT_USER" 2>&1 | grep -v "usermod: no changes" || true
        fi

        # Copy the pam_apparmor profiles and restart apparmor.
        if [ -d "${WG_SHAREDIR}/wg_system/apparmor-profiles" ] ; then
            if [ "${1}" = "install" ] ; then
                echo "Setting up pam_apparmor profiles"
            else
                echo "Updating pam_apparmor profiles"
            fi

            # Create our variables file
            mkdir -p /etc/apparmor.d/tunables
            /bin/rm -f /etc/apparmor.d/tunables/wg_system >/dev/null || true
            cat > /etc/apparmor.d/tunables/wg_system << CTRL_EOF
@{WG_ETCDIR}=${WG_ETCDIR}/
@{WG_BINDIR}=${WG_BINDIR}/
@{WG_LIBDIR}=${WG_LIBDIR}/
@{WG_VARDIR}=${WG_VARDIR}/
@{WG_RUNDIR}=${WG_RUNDIR}/
CTRL_EOF
            chmod 0644 /etc/apparmor.d/tunables/wg_system 

            if [ -d "${WG_SHAREDIR}/wg_system/apparmor-profiles/pam" ] ; then
                mkdir -p /etc/apparmor.d/pam
                cp -f ${WG_SHAREDIR}/wg_system/apparmor-profiles/pam/* /etc/apparmor.d/pam
            fi
            cp -f ${WG_SHAREDIR}/wg_system/apparmor-profiles/pam_* /etc/apparmor.d

            # Get AppArmor to reload profiles.
            if [ "${1}" = "upgrade" ] && [ -x /etc/init.d/apparmor ] ; then
                if [ -x /sbin/initctl ]; then
                /etc/init.d/apparmor restart || true
                fi
                if [ -x /usr/bin/systemctl ]; then
                    /usr/bin/systemctl reload apparmor
                    /usr/bin/systemctl restart apparmor
                fi
            fi
        fi

        # Copy the pam profiles.
        if [ "${1}" = "install" ] ; then
            echo "Setting up pam profiles"
        else
            echo "Updating pam profiles"
        fi

    for file in /etc/pam.d/login /etc/pam.d/sshd ; do
        tmpfile=`mktemp`

            # The following sed command is a work-around for an issue
            # encountered in Ubuntu 14.04 LTS testing causing sshd
            # logins to fail and resulting in /var/log/auth.log
            # messages such as "error: PAM: pam_open_session(): Cannot
            # make/remove an entry for the specified session" and
            # "System error"
            sed -e 's/^\(session    required     pam_loginuid.so\)$/\# \1/' $file >$tmpfile

            if ! grep -q pam_apparmor.so $tmpfile ; then
                cat <<EOF >>$tmpfile

session    optional     pam_apparmor.so order=user,group,default
EOF
            fi

            if /usr/bin/cmp -s $file $tmpfile ; then
                /bin/rm -f $tmpfile
            else
                /bin/mv -f $tmpfile $file
                chmod 644 $file
            fi
        done

        # Setup the motd update 
        if [ -x "${WG_BINDIR}/wg_system_motd.sh" ] ; then
            WG_SYSTEM_UPDATE_MOTD=/etc/update-motd.d/50-wg-system
            /bin/rm -f ${WG_SYSTEM_UPDATE_MOTD} >/dev/null || true
            ln -sf "${WG_BINDIR}/wg_system_motd.sh" ${WG_SYSTEM_UPDATE_MOTD}
        fi

        # Remove obsolete upstart scripts from /etc/init
        WG_PRODUCT_CAPITALIZED=`echo $WG_PRODUCT | sed 's/./\U&/'`
        /bin/rm -f /etc/init/${WG_PRODUCT_CAPITALIZED}-resize-datadisk.conf

        if [ -e /etc/default/wg_azure ]; then
            echo "File /etc/default/wg_azure exists"
        else
            echo "No file /etc/default/wg_azure"
        fi

        if [ -x /sbin/initctl ]; then
        # Copy the upstart scripts to /etc/init
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_apttoken.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_apttoken.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_azure_resource.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_azure_resource.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_azure_state.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_azure_state.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_backup.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_backup.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_callhome.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_callhome.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_data_disk_resize.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_data_disk_resize.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_db_analyze.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_db_analyze.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_event.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_event.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_get_feature_keys.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_get_feature_keys.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_systemd.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_systemd.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_system_init.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_system_init.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_system_reset.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_system_reset.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wg_vm_detect.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wg_vm_detect.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wgdevice_ssh.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wgdevice_ssh.conf
        cp -fp ${WG_SHAREDIR}/wg_system/init/wgdevice_ssh_init.conf /etc/init
        chown -R --reference=/etc/init /etc/init/wgdevice_ssh_init.conf
        fi

        if [ -x /usr/bin/systemctl ]; then
            # Copy the systemd service units to /etc/systemd/system
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wg-systemd-event-handler.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wg-systemd-event-handler.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wg-apttoken.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wg-apttoken.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wg-backup.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wg-backup.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wg-callhome.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wg-callhome.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wg-data-disk-resize.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wg-data-disk-resize.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wg-db-analyze.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wg-db-analyze.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wgdevice-ssh-init.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wgdevice-ssh-init.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wgdevice-ssh.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wgdevice-ssh.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wg-get-feature-keys.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wg-get-feature-keys.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wg-ntpd.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wg-ntpd.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wg-resolve.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wg-resolve.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wg-routes.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wg-routes.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wg-sshd.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wg-sshd.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wg-systemd.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wg-systemd.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wg-system-init.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wg-system-init.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wg-system-reset.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wg-system-reset.service
            cp -fp ${WG_SHAREDIR}/wg_system/systemd/wg-vm-detect.service ${WG_SYSTEMDDIR}
            chown -R --reference=${WG_SYSTEMDDIR} ${WG_SYSTEMDDIR}/wg-vm-detect.service
            # Enable some systemd service units that need to be started at bootup stage
            # NOTE: We should not use "systemctl enable" command to enable these service units since systemd is not running in chroot environment.
            ln -sf ${WG_SYSTEMDDIR}/wg-systemd-event-handler.service ${WG_SYSTEMDDIR}/multi-user.target.wants/wg-systemd-event-handler.service
            ln -sf ${WG_SYSTEMDDIR}/wg-apttoken.service ${WG_SYSTEMDDIR}/multi-user.target.wants/wg-apttoken.service
            ln -sf ${WG_SYSTEMDDIR}/wg-backup.service ${WG_SYSTEMDDIR}/multi-user.target.wants/wg-backup.service
            ln -sf ${WG_SYSTEMDDIR}/wg-callhome.service ${WG_SYSTEMDDIR}/multi-user.target.wants/wg-callhome.service
            ln -sf ${WG_SYSTEMDDIR}/wg-data-disk-resize.service ${WG_SYSTEMDDIR}/multi-user.target.wants/wg-data-disk-resize.service
            ln -sf ${WG_SYSTEMDDIR}/wgdevice-ssh-init.service ${WG_SYSTEMDDIR}/multi-user.target.wants/wgdevice-ssh-init.service
            ln -sf ${WG_SYSTEMDDIR}/wgdevice-ssh.service ${WG_SYSTEMDDIR}/multi-user.target.wants/wgdevice-ssh.service
            ln -sf ${WG_SYSTEMDDIR}/wg-get-feature-keys.service ${WG_SYSTEMDDIR}/multi-user.target.wants/wg-get-feature-keys.service
            ln -sf ${WG_SYSTEMDDIR}/wg-ntpd.service ${WG_SYSTEMDDIR}/multi-user.target.wants/wg-ntpd.service
            ln -sf ${WG_SYSTEMDDIR}/wg-resolve.service ${WG_SYSTEMDDIR}/multi-user.target.wants/wg-resolve.service
            ln -sf ${WG_SYSTEMDDIR}/wg-routes.service ${WG_SYSTEMDDIR}/multi-user.target.wants/wg-routes.service
            ln -sf ${WG_SYSTEMDDIR}/wg-sshd.service ${WG_SYSTEMDDIR}/multi-user.target.wants/wg-sshd.service
            ln -sf ${WG_SYSTEMDDIR}/wg-systemd.service ${WG_SYSTEMDDIR}/multi-user.target.wants/wg-systemd.service
            ln -sf ${WG_SYSTEMDDIR}/wg-system-init.service ${WG_SYSTEMDDIR}/multi-user.target.wants/wg-system-init.service
            ln -sf ${WG_SYSTEMDDIR}/wg-vm-detect.service ${WG_SYSTEMDDIR}/multi-user.target.wants/wg-vm-detect.service
            # Copy D-Bus configuration file
            cp -fp ${WG_SHAREDIR}/wg_system/conf/com.watchguard.system.conf /etc/dbus-1/system.d
        fi

        # Copy the dhclient hook scripts
        if [ -d /etc/dhcp/dhclient-enter-hooks.d ] ; then
            cp ${WG_SHAREDIR}/wg_system/dhclient/wg_dhcp_enter_hook /etc/dhcp/dhclient-enter-hooks.d
        fi
        if [ -d /etc/dhcp/dhclient-exit-hooks.d ] ; then
            cp ${WG_SHAREDIR}/wg_system/dhclient/wg_dhcp_exit_hook /etc/dhcp/dhclient-exit-hooks.d
        fi

        # Copy the postinst.d files
        if [ -d /etc/kernel/postinst.d ]; then
            cp ${WG_SHAREDIR}/wg_system/kernel/postinst.d/zz-extlinux /etc/kernel/postinst.d/
        fi

        # Update extlinux.conf, if necessary
        WG_EXTLINUX_CONF=/boot/extlinux/extlinux.conf
        if [ -f ${WG_EXTLINUX_CONF} ]; then
            cat ${WG_EXTLINUX_CONF} |sed -e '
            /APPEND .*wgrd.root ro$/ s/$/ console=ttyS0 console=tty0 earlyprintk=ttyS0/
            ' > ${WG_EXTLINUX_CONF}.new
            if  diff -q ${WG_EXTLINUX_CONF} ${WG_EXTLINUX_CONF}.new >/dev/null ; then
                # no change, do nothing
                rm -f ${WG_EXTLINUX_CONF}.new
            else
                cp -f ${WG_EXTLINUX_CONF} ${WG_EXTLINUX_CONF}.bak
                mv -f ${WG_EXTLINUX_CONF}.new ${WG_EXTLINUX_CONF}
                chmod 0644 "${WG_EXTLINUX_CONF}"
            fi
        fi

        # Copy the default configuration files.
        mkdir -p ${WG_ETCDIR}/system/defaults
        cp -fp ${WG_SHAREDIR}/wg_system/defaults/*.json ${WG_ETCDIR}/system/defaults
        chmod 664 ${WG_ETCDIR}/system/defaults/*.json

        # Install the logrotate configuration file for rsyslog.
        cp -f ${WG_SHAREDIR}/wg_system/defaults/wg_logrotate_rsyslog /etc/logrotate.d/rsyslog
        chmod 644 /etc/logrotate.d/rsyslog

        # Install the configuration file for device SSH service.
        WGDEVICE_SSHDIR="${WG_ETCDIR}/wgdevice_ssh"
        DEFAULT_CONFIG="${WG_SHAREDIR}/wg_system/defaults/wgdevice_sshd_config"
        SSHD_CONFIG="${WGDEVICE_SSHDIR}/sshd_config"
        if [ ! -d "$WGDEVICE_SSHDIR" ] ; then
            mkdir -p -m0755 "$WGDEVICE_SSHDIR"
            chown  --reference="$WG_ETCDIR" "$WGDEVICE_SSHDIR"
        fi
        if [ -f "$DEFAULT_CONFIG" ] ; then
            cp -f $DEFAULT_CONFIG $SSHD_CONFIG
            chmod 644 $SSHD_CONFIG
            chown --reference="$WGDEVICE_SSHDIR" $SSHD_CONFIG
        fi

        # Install an hourly CRON job to run logrotate
        if [ ! -e /etc/cron.hourly/logrotate ]; then
            /bin/cp -f /etc/cron.daily/logrotate /etc/cron.hourly
        fi

        # Install the systemd-journald configuration file
        if [ -x /usr/bin/systemctl ]; then
            cp -f ${WG_SHAREDIR}/wg_system/defaults/wg_systemd_journald.conf /etc/systemd/journald.conf
            chmod 644 /etc/systemd/journald.conf
            if [ "${1}" = "upgrade" ]; then
                /usr/bin/systemctl restart systemd-journald
            fi
        fi

        if [ "${1}" = "upgrade" ] ; then
            if [ ! -f ${WG_ETCDIR}/system/system.json ] ; then
                # Upgrading a system in factory-default state.
                cp -fp ${WG_ETCDIR}/system/defaults/* ${WG_ETCDIR}/system
            else
                # Copy new/missing configuration files from default directory
                for name in $(find ${WG_ETCDIR}/system/defaults/ -type f -printf "%f ")
                do
                    if [ ! -f ${WG_ETCDIR}/system/$name ] ; then
                        cp -fp ${WG_ETCDIR}/system/defaults/$name ${WG_ETCDIR}/system
                        if [ $name = "postfix.json" ] ; then
                            migrate_smtp_config
                        fi
                    fi
                done
                reconfig_postfix
            fi
        fi

        chown -R --reference=${WG_ETCDIR} ${WG_ETCDIR}/system

        # Create backup config directory and mount point.
        if [ ! -d ${WG_ETCDIR}/backup ] ; then
            mkdir -p ${WG_ETCDIR}/backup
            chown -R --reference=${WG_ETCDIR} ${WG_ETCDIR}/backup
            chmod 775 ${WG_ETCDIR}/backup
        fi

        if /bin/mount | grep -q "${WG_VARDIR}/backup_mnt\>" ; then
            # Currently mounted on, so it exists.
            :
        elif [ ! -d ${WG_VARDIR}/backup_mnt ] ; then
            mkdir -p ${WG_VARDIR}/backup_mnt
            chown -R --reference=${WG_VARDIR} ${WG_VARDIR}/backup_mnt
            chmod 775 ${WG_VARDIR}/backup_mnt
        fi

        # Install/update sshd trusted CA key
        cp -fp ${WG_SHAREDIR}/wg_system/defaults/sshd_ca_key.pub /etc/ssh/sshd_ca_keys
        chown --reference=/etc/ssh /etc/ssh/sshd_ca_keys
        chmod 644 /etc/ssh/sshd_ca_keys

        # Make sure all moduli in /etc/ssh/moduli are at least 2048 bits.
        remove_weak_ssh_moduli

        # Make sure wgadmin is in the sudoers list.
        if [ -f /etc/sudoers.d/wg_${WG_PRODUCT} ] ; then
            mv /etc/sudoers.d/wg_${WG_PRODUCT} /etc/sudoers.d/wg_${WG_PRODUCT}~
        fi
        tmpf=`mktemp`
        cat <<EOF >"$tmpf"
# wgadmin user is the default user in ${WG_PRODUCT} images.
# This user needs passwordless sudo functionality.
#
# NOTE: THIS FILE IS AUTOMATICALLY GENERATED.
#    ALL EDITS TO THIS FILE WILL BE LOST ON ${WG_PRODUCT} UPGRADES.
${WG_ADMIN_USER} ALL=(ALL) NOPASSWD:ALL
EOF
        /bin/mv -f "$tmpf" /etc/sudoers.d/wg_${WG_PRODUCT}
        chmod 440 /etc/sudoers.d/wg_${WG_PRODUCT}

        # Create the task directories
        mkdir -p ${WG_VARDIR}/system/task-requests
        mkdir -p ${WG_VARDIR}/system/task-status
        chown -R --reference=${WG_VARDIR} ${WG_VARDIR}/system

        # Create the upload directory
        mkdir -p ${WG_VARDIR}/upload
        chown -R --reference=${WG_VARDIR} ${WG_VARDIR}/upload

        # Install daily cron job to retrieve APT token and remove
        # expired files in $WG_VARDIR

        # NOTE: according to run-parts(8), the cron job file name
        # "must consist entirely of ASCII upper- and lower-case
        # letters, ASCII digits, ASCII underscores, and ASCII
        # minus-hyphens."  No other punctuation is allowed.

        tmpf=`mktemp`
        cat <<EOF >"$tmpf"
#!/bin/sh
# ${WG_PRODUCT} daily cron job
#
# NOTE: THIS FILE IS AUTOMATICALLY GENERATED.
#    ALL EDITS TO THIS FILE WILL BE LOST ON ${WG_PRODUCT} UPGRADES.

set -u

DEFAULT=/etc/default/wg_system

# read our config or else quit
test -r \$DEFAULT || exit 0
. \$DEFAULT || exit 1

# check integrity of downloaded package lists and remove corrupted
# list files
\$WG_BINDIR/wg_apt_update.sh -n

# clean up downloaded package files
apt-get clean

# purge package apt-xapian-index
dpkg --purge apt-xapian-index || true

# check and, if needed, try to get an APT access token
if [ -x /sbin/initctl ]; then
/sbin/initctl start --no-wait wg_apttoken || true
fi
if [ -x /usr/bin/systemctl ]; then
    /usr/bin/systemctl restart wg-apttoken.service
fi

# check and, if needed, try to get appliance feature keys
if [ -x /sbin/initctl ]; then
/sbin/initctl start --no-wait wg_get_feature_keys || true
fi
if [ -x /usr/bin/systemctl ]; then
    /usr/bin/systemctl restart wg-get-feature-keys.service
fi

# Analyze yesterday's segment tables in the database
if [ -x /sbin/initctl ]; then
/sbin/initctl emit --no-wait wg_event_db_analyze
fi
if [ -x /usr/bin/systemctl ]; then
    /usr/bin/systemctl restart wg-db-analyze.service
fi

# change to \$WG_VARDIR directory
cd \$WG_VARDIR || exit 1

# find and remove log files modified 30 days ago or prior
find logs -type f -mtime +29 -print0 | xargs -0 /bin/rm -f

# find and remove system task request files modified 30 days ago or
# prior
find system/task-requests -type f -mtime +29 -print0 | xargs -0 /bin/rm -f

# find and remove system task status files modified 30 days ago or
# prior
find system/task-status -type f -mtime +29 -print0 | xargs -0 /bin/rm -f

# find and remove upload files modified one day ago or prior
find upload -type f -mtime +0 -print0 | xargs -0 /bin/rm -f

# find and remove temporary files modified one day ago or prior
if [ -d data/tmp ] ; then
    find data/tmp -type f -mtime +0 -print0 | xargs -0 /bin/rm -f
fi

# find and remove session files modified one day ago or prior
if [ -d data/session ] ; then
    find data/session -type f -mtime +0 -print0 | xargs -0 /bin/rm -f
fi

# check if the wsserver certificate is expired
\$WG_BINDIR/python \$WG_BINDIR/wg_regenerate_cert.pyc
EOF
        /bin/mv -f "$tmpf" /etc/cron.daily/wg_${WG_PRODUCT}
        chmod 755 /etc/cron.daily/wg_${WG_PRODUCT}

        # Install hourly cron job to clean up PostgreSQL log files
        
        # NOTE: according to run-parts(8), the cron job file name
        # "must consist entirely of ASCII upper- and lower-case
        # letters, ASCII digits, ASCII underscores, and ASCII
        # minus-hyphens."  No other punctuation is allowed.

        /bin/rm -f /etc/cron.hourly/wg_clean_psql_logs
        /bin/cp -f ${WG_BINDIR}/wg_clean_psql_logs.sh /etc/cron.hourly/wg_clean_psql_logs
        
        mkdir -p "${WG_APTDIR}"

        # SACK Panic and Other TCP Denial of Service Issues
        # https://wiki.ubuntu.com/SecurityTeam/KnowledgeBase/SACKPanic
        tmpf=`mktemp`
        cat <<EOF >"$tmpf"
# Turn off TCP MTU probing
net.ipv4.tcp_mtu_probing=0
EOF
        /bin/mv -f "$tmpf" /etc/sysctl.d/99-wg_${WG_PRODUCT}.conf
        chmod 0644 /etc/sysctl.d/99-wg_${WG_PRODUCT}.conf
        sysctl -w net.ipv4.tcp_mtu_probing=0

        # Update 50unattended-upgrades to remove unused dependencies after an unattended upgrade.
        # More details, see https://help.ubuntu.com/community/RemoveOldKernels
        rc=0
        cat /etc/apt/apt.conf.d/50unattended-upgrades | egrep '^\/\/Unattended-Upgrade::Remove-Unused-Dependencies "false"' > /dev/null || rc=$?
        if [ $rc -eq 0 ]; then
            sed -i 's/^\/\/Unattended-Upgrade::Remove-Unused-Dependencies "false"/Unattended-Upgrade::Remove-Unused-Dependencies "true"/g' /etc/apt/apt.conf.d/50unattended-upgrades
        fi

        # Update 50unattended-upgrades to disable libnetplan0 and netplan.io packages.
        rc=0
        cat /etc/apt/apt.conf.d/50unattended-upgrades | egrep '^    "libnetplan0";' > /dev/null || rc=$?
        if [ $rc -ne 0 ]; then
            sed -i '/^Unattended-Upgrade::Package-Blacklist {/a\    "libnetplan0";' /etc/apt/apt.conf.d/50unattended-upgrades
        fi

        rc=0
        cat /etc/apt/apt.conf.d/50unattended-upgrades | egrep '^    "netplan.io";' > /dev/null || rc=$?
        if [ $rc -ne 0 ]; then
            sed -i '/^Unattended-Upgrade::Package-Blacklist {/a\    "netplan.io";' /etc/apt/apt.conf.d/50unattended-upgrades
        fi

        # Disable floppy driver and update initramfs
        if [ ! -f /etc/modprobe.d/blacklist-floppy.conf ]; then
            echo "blacklist floppy" > /etc/modprobe.d/blacklist-floppy.conf || true
            /usr/sbin/update-initramfs -u || true
        fi

        if [ "${1}" = "upgrade" ] ; then

            if /sbin/lsmod | grep -q "floppy" ; then
                /usr/sbin/rmmod floppy || true
            fi

            if [ -d ${WG_VARDIR}/data ] ; then
                # Create the tmp and session directories.
                mkdir -p ${WG_VARDIR}/data/tmp
                chown -R --reference=${WG_VARDIR} ${WG_VARDIR}/data/tmp
                chmod -R --reference=${WG_VARDIR} ${WG_VARDIR}/data/tmp

                mkdir -p ${WG_VARDIR}/data/session
                chown -R --reference=${WG_VARDIR} ${WG_VARDIR}/data/session
                chmod -R --reference=${WG_VARDIR} ${WG_VARDIR}/data/session
                rm -f ${WG_VARDIR}/data/session/*
            fi

            # Upgrade Microsoft Azure Linux Agent
            ${WG_BINDIR}/wg_azure_agent.sh ${1}
            

            # For systemd we should run "systemctl daemon-reload" command to reload the new service units.
            if [ -x /usr/bin/systemctl ]; then
                /usr/bin/systemctl daemon-reload >/dev/null || true
            fi

            # For systemd we should start wg-systemd-event-handler at first since the upstart profile wg_event.conf is no longer existed.
            if [ -x /usr/bin/systemctl ]; then
                echo "Starting wg-systemd-event-handler"
                /usr/bin/systemctl start wg-systemd-event-handler >/dev/null || true
            fi

            # Start wg_systemd, but only on upgrades.
            echo "Starting wg_systemd"
            if [ -x /sbin/initctl ]; then
            /sbin/initctl start wg_systemd >/dev/null || true
            fi
            if [ -x /usr/bin/systemctl ]; then
                /usr/bin/systemctl start wg-systemd >/dev/null || true
            fi

            # Start the wg_backup upstart process. Don't wait for it to
            # start though since this could potentially take a long time.
            echo "Starting wg_backup"
            if [ -x /sbin/initctl ]; then
            /sbin/initctl start --no-wait wg_backup >/dev/null || true
            fi
            if [ -x /usr/bin/systemctl ]; then
                /usr/bin/systemctl start wg-backup >/dev/null || true
            fi

            # Start the callhome service
            echo "Starting wg_callhome"
            if [ -x /sbin/initctl ]; then
            /sbin/initctl start wg_callhome >/dev/null || true
            fi
            if [ -x /usr/bin/systemctl ]; then
                /usr/bin/systemctl start wg-callhome >/dev/null || true
            fi

            # Try to retrieve an APT token immediately on upgrades. If this
            # fails the daily cronjob will keep trying.
            echo "Starting wg_apttoken"
            if [ -x /sbin/initctl ]; then
            /sbin/initctl start --no-wait wg_apttoken >/dev/null || true
            fi
            if [ -x /usr/bin/systemctl ]; then
                /usr/bin/systemctl start wg-apttoken >/dev/null || true
            fi

            # Redetect virtualization platform
            echo "Starting wg_vm_detect"
            if [ -x /sbin/initctl ]; then
            /sbin/initctl start --no-wait wg_vm_detect >/dev/null || true
            fi
            if [ -x /usr/bin/systemctl ]; then
                /usr/bin/systemctl start wg-vm-detect >/dev/null || true
            fi

            # Start the wgdevice_ssh server
            echo "Starting wgdevice SSH server"
            if [ -x /sbin/initctl ]; then
            /sbin/initctl start --no-wait wgdevice_ssh >/dev/null || true
            fi
            if [ -x /usr/bin/systemctl ]; then
                /usr/bin/systemctl start wgdevice-ssh-init >/dev/null || true
                /usr/bin/systemctl start wgdevice-ssh >/dev/null || true
            fi

            # Check and regenerate /etc/ssh/sshd_config, if needed. Note:
            # This should be done after wg_systemd has been started.
            check_sshd_config

            # Clean up any local log archives accidentally enabled by 1.0
            # users. These just consume space and are not accessible to
            # users.
            /bin/rm -f ${WG_VARDIR}/wlogserver/backup/*.zip

            # Cleanup APT archives
            apt-get clean || true

            # Update APT configuration and keyring
            update_apt_keyring

            # Upgrade /etc/fstab, setting <pass> to 1 or 2, depending upon the
            # file mount point.  A <mount point> of "/" sets <pass> to 1, a
            # <mount point> "/.+" sets <pass> to 2.

            tmpf=`mktemp`
            awk -f - /etc/fstab <<EOF >"$tmpf"
BEGIN {
    FS = "[ \t]+";
    format = "%-21s %-35s %-7s %-24s %-7s %-7s\n";
}
/^# <file system>/ {
    printf(format, "# <file system>", "<mount point>", "<type>", "<options>", "<dump>", "<pass>");
    next;
}
/^#/ {
    print \$0;
    next;
}
/^$/ {
    print \$0;
    next;
}
{
    pass = 0;
    if ( match(\$3, "^ext[234]\$") )
        if ( \$2 == "/" )
            pass = 1;
        else if ( match(\$2, "^/[^/]+") )
            pass = 2;
    printf(format, \$1, \$2, \$3, \$4, \$5, pass);
}
EOF
            /bin/mv -f "$tmpf" /etc/fstab
            chmod 644 /etc/fstab

            # Upgrade /etc/default/rcS, setting FSCKFIX=yes.

            tmpf=`mktemp`
            sed -e 's/^FSCKFIX=no$/FSCKFIX=yes/' /etc/default/rcS >"$tmpf"
            /bin/mv -f "$tmpf" /etc/default/rcS
            chmod 644 /etc/default/rcS

            if [ ! -e /etc/default/wg_azure ]; then
                # Check for the iptables rule that blocks the wgsupport user from connecting to the Log Server
                rc=0
                iptables -C OUTPUT -p tcp -d 127.0.0.1 --dport 4121 -m owner --uid-owner $WG_SUPPORT_USER -j REJECT 2> /dev/null || rc=$?
                if [ $rc -ne 0 ]; then
                    rc=0
                    iptables -I OUTPUT -p tcp -d 127.0.0.1 --dport 4121 -m owner --uid-owner $WG_SUPPORT_USER -j REJECT || rc=$?
                    if [ $rc -eq 0 ]; then
                        echo "Added $WG_SUPPORT_USER system account iptables rules"
                        tmpf=`mktemp`
                        rc=0
                        iptables-save -c > "$tmpf" || rc=$?
                        if [ $rc -eq 0 ]; then
                            if [ -e /etc/iptables.rules ]; then
                                /bin/mv -f /etc/iptables.rules /etc/iptables.rules.bak || true
                            fi
                            /bin/mv -f "$tmpf" /etc/iptables.rules || true
                            chmod 0644 /etc/iptables.rules || true
                        fi
                    else
                        echo "Warning: unable to add $WG_SUPPORT_USER system account iptables rules"
                    fi
                fi
            fi

            # Mitigates CVE-2019-11477, CVE-2019-11478 and CVE-2019-11479 by preventing new connections made with low MSS sizes
            rc=0
            iptables -C INPUT -p tcp -m tcpmss --mss 1:500 -j DROP 2> /dev/null || rc=$?
            if [ $rc -ne 0 ]; then
                rc=0
                iptables -I INPUT -p tcp -m tcpmss --mss 1:500 -j DROP || rc=$? 
                if [ $rc -eq 0 ]; then
                    echo "Added iptables rules to solve Linux kernel SACK Panic issue"
                    tmpf=`mktemp`
                    rc=0
                    iptables-save -c > "$tmpf" || rc=$?
                    if [ $rc -eq 0 ]; then
                        if [ -e /etc/iptables.rules ]; then
                            /bin/mv -f /etc/iptables.rules /etc/iptables.rules.bak || true
                        fi
                        /bin/mv -f "$tmpf" /etc/iptables.rules || true
                        chmod 0644 /etc/iptables.rules || true
                    fi
                else
                    echo "Warning: unable to add iptables rules to solve Linux kernel SACK Panic issue"
                fi
            fi

            # Suppress iptables denied logs
            rc=0
            iptables -C INPUT -m limit --limit 5/min -j LOG --log-prefix "iptables denied: " --log-level 7 2> /dev/null || rc=$?
            if [ $rc -eq 0 ]; then
                rc=0
                iptables -D INPUT -m limit --limit 5/min -j LOG --log-prefix "iptables denied: " --log-level 7 2> /dev/null || rc=$?
                if [ $rc -eq 0 ]; then
                    echo "Remove iptables denied rule"
                    tmpf=`mktemp`
                    rc=0
                    iptables-save -c > "$tmpf" || rc=$?
                    if [ $rc -eq 0 ]; then
                        if [ -e /etc/iptables.rules ]; then
                            /bin/mv -f /etc/iptables.rules /etc/iptables.rules.bak || true
                        fi
                        /bin/mv -f "$tmpf" /etc/iptables.rules || true
                        chmod 0644 /etc/iptables.rules || true
                    fi
                else
                    echo "Warning: unable to remove iptables denied rule"
                fi
            fi

            # Fix FBX-27659
            tmpf=`mktemp`
            cat << EOF > "$tmpf"
#!/bin/sh
locks=\$(lslocks | grep "/var/lib/dpkg/lock")
while [ -n "\$locks" ]; do
    sleep 1
    locks=\$(lslocks | grep "/var/lib/dpkg/lock")
done
libver=\$(dpkg-query --showformat='\${Version}' --show libnetplan0)
appver=\$(dpkg-query --showformat='\${Version}' --show netplan.io)
if [ "\${libver}" != "\${appver}" ]; then
    apt-get install libnetplan0=\$appver -y --allow-downgrades
fi
holds=\$(apt-mark showhold | grep -e "libnetplan0" -e "netplan.io")
if [ -z "\$holds" ]; then
    apt-mark hold libnetplan0
    apt-mark hold netplan.io
fi
EOF
            /usr/bin/systemd-run --on-active=5 /bin/sh "$tmpf" || true

        fi

        ;;

    "uninstall")

        if [ -r /etc/default/wg_system ] ; then
            . /etc/default/wg_system
        else
            # Nothing to do
            exit 0
        fi

        if [ -f ${WG_SYSTEMDDIR}/wg-systemd-event-handler.service ]; then
            echo "Stopping wg-systemd-event-handler"
            /usr/bin/systemctl stop wg-systemd-event-handler >/dev/null || true
            /usr/bin/systemctl disable wg-systemd-event-handler >/dev/null || true
            /bin/rm -f ${WG_SYSTEMDDIR}/wg-systemd-event-handler.service
        fi

        if [ -f /etc/init/wg_systemd.conf ] ; then
            echo "Stopping wg_systemd"
            /sbin/initctl stop wg_systemd >/dev/null || true
            /bin/rm -f /etc/init/wg_systemd.conf
        fi
        if [ -f ${WG_SYSTEMDDIR}/wg-systemd.service ] ; then
            echo "Stopping wg_systemd"
            /usr/bin/systemctl stop wg-systemd >/dev/null || true
            /usr/bin/systemctl disable wg-systemd >/dev/null || true
            /bin/rm -f ${WG_SYSTEMDDIR}/wg-systemd.service
        fi

        if [ -f /etc/init/wg_backup.conf ] ; then
            echo "Stopping wg_backup"
            /sbin/initctl stop wg_backup >/dev/null || true
            /bin/rm -f /etc/init/wg_backup.conf
        fi
        if [ -f ${WG_SYSTEMDDIR}/wg-backup.service ] ; then
            echo "Stopping wg_backup"
            /usr/bin/systemctl stop wg-backup >/dev/null || true
            /usr/bin/systemctl disable wg-backup >/dev/null || true
            /bin/rm -f ${WG_SYSTEMDDIR}/wg-backup.service
        fi

        if [ -f /etc/init/wgdevice_ssh.conf ] ; then
            echo "Stopping wgdevice SSH server"
            /sbin/initctl stop wgdevice_ssh >/dev/null || true
            /bin/rm -f /etc/init/wgdevice_ssh.conf
            /bin/rm -f /etc/init/wgdevice_ssh_init.conf
        fi
        if [ -f ${WG_SYSTEMDDIR}/wgdevice-ssh.service ] ; then
            echo "Stopping wgdevice SSH server"
            /usr/bin/systemctl stop wgdevice-ssh >/dev/null || true
            /usr/bin/systemctl disable wgdevice-ssh >/dev/null || true
            /bin/rm -f ${WG_SYSTEMDDIR}/wgdevice-ssh.service
            /usr/bin/systemctl stop wgdevice-ssh-init >/dev/null || true
            /usr/bin/systemctl disable wgdevice-ssh-init >/dev/null || true
            /bin/rm -f ${WG_SYSTEMDDIR}/wgdevice-ssh-init.service
        fi

        if [ -x /sbin/initctl ]; then
        # remove upstart profiles
        /bin/rm -f /etc/init/wg_apttoken.conf
        /bin/rm -f /etc/init/wg_azure_resource.conf
        /bin/rm -f /etc/init/wg_azure_state.conf
        /bin/rm -f /etc/init/wg_callhome.conf
        /bin/rm -f /etc/init/wg_data_disk_resize.conf
        /bin/rm -f /etc/init/wg_db_analyze.conf
        /bin/rm -f /etc/init/wg_event.conf
        /bin/rm -f /etc/init/wg_get_feature_keys.conf
        /bin/rm -f /etc/init/wg_system_init.conf
        /bin/rm -f /etc/init/wg_system_reset.conf
        /bin/rm -f /etc/init/wg_vm_detect.conf
        fi

        if [ -x /usr/bin/systemctl ]; then
            # Disable systemd service units
            /usr/bin/systemctl disable wg-apttoken >/dev/null || true
            /usr/bin/systemctl disable wg-callhome >/dev/null || true
            /usr/bin/systemctl disable wg-data-disk-resize >/dev/null || true
            /usr/bin/systemctl disable wg-db-analyze >/dev/null || true
            /usr/bin/systemctl disable wg-get-feature-keys >/dev/null || true
            /usr/bin/systemctl disable wg-ntpd >/dev/null || true
            /usr/bin/systemctl disable wg-resolve >/dev/null || true
            /usr/bin/systemctl disable wg-sshd >/dev/null || true
            /usr/bin/systemctl disable wg-system-init >/dev/null || true
            /usr/bin/systemctl disable wg-system-reset >/dev/null || true
            /usr/bin/systemctl disable wg-vm-detect >/dev/null || true
            # Remove systemd service units
            /bin/rm -f ${WG_SYSTEMDDIR}/wg-apttoken.service
            /bin/rm -f ${WG_SYSTEMDDIR}/wg-callhome.service
            /bin/rm -f ${WG_SYSTEMDDIR}/wg-data-disk-resize.service
            /bin/rm -f ${WG_SYSTEMDDIR}/wg-db-analyze.service
            /bin/rm -f ${WG_SYSTEMDDIR}/wg-get-feature-keys.service
            /bin/rm -f ${WG_SYSTEMDDIR}/wg-ntpd.service
            /bin/rm -f ${WG_SYSTEMDDIR}/wg-resolve.service
            /bin/rm -f ${WG_SYSTEMDDIR}/wg-routes.service
            /bin/rm -f ${WG_SYSTEMDDIR}/wg-sshd.service
            /bin/rm -f ${WG_SYSTEMDDIR}/wg-system-init.service
            /bin/rm -f ${WG_SYSTEMDDIR}/wg-system-reset.service
            /bin/rm -f ${WG_SYSTEMDDIR}/wg-vm-detect.service
            # After we remove all of Dimension specified service units, we should reload the rest of service units.
            /usr/bin/systemctl daemon-reload >/dev/null || true
        fi

        # remove callhome CRON job
        /bin/rm -f /var/spool/cron/crontabs/${WG_ADMIN_USER}

        # remove daily CRON jobs
        /bin/rm -f /etc/cron.daily/wg_${WG_PRODUCT}

        # remove hourly CRON jobs
        /bin/rm -f /etc/cron.hourly/wg_clean_psql_logs

        # remove MOTD customizations
        /bin/rm -f /etc/update-motd.d/50-wg-system

        ;;

    *)

        usage $0

        ;;

esac

exit 0
