-- This file contains the SQL statements that create the database
-- objects used by the Dimension visibility module.

SET SEARCH_PATH TO public;
CREATE TABLE g_visibility_config(
    dynamic_host_lookup_flag INTEGER NOT NULL DEFAULT 0,
    add_new_devices          INTEGER NOT NULL DEFAULT 1,
    accept_diag_logs         INTEGER NOT NULL DEFAULT 1,
    anonymize_reports        BOOLEAN NOT NULL DEFAULT FALSE,
    anonymize_title          TEXT DEFAULT '',
    anonymize_desc           TEXT DEFAULT '',
    session_auditing         INTEGER DEFAULT 0,
    purge_diagnostic_log     BOOLEAN DEFAULT FALSE
);
INSERT INTO g_visibility_config (dynamic_host_lookup_flag,
    add_new_devices, accept_diag_logs, anonymize_reports,
    anonymize_title, anonymize_desc, session_auditing)
    SELECT 0, 1, 1, False, '', '', 0 WHERE NOT EXISTS (SELECT * FROM g_visibility_config LIMIT 1);

CREATE TABLE g_visibility_clients(
   appliance_id      INTEGER NOT NULL PRIMARY KEY,
   accept_logs       INTEGER NOT NULL,
   accept_diag_logs  INTEGER NOT NULL DEFAULT -1,
   expiry_time       TIMESTAMP);

CREATE SEQUENCE cluster_traffic_sid_seq MINVALUE 1 MAXVALUE 2147483647 CYCLE;
CREATE TABLE cluster_traffic(
   sid            INTEGER DEFAULT NEXTVAL('cluster_traffic_sid_seq'), 
   cluster        TEXT DEFAULT '', 
   sn             TEXT NOT NULL, 
   msg_id         TEXT,
   tag_id         INTEGER, 
   raw_id         BIGINT DEFAULT 0, 
   disp           SMALLINT, 
   direction      SMALLINT, 
   pri            SMALLINT, 
   policy         TEXT, 
   protocol       TEXT, 
   src_ip         TEXT, 
   src_port       INTEGER, 
   dst_ip         TEXT, 
   dst_port       INTEGER, 
   src_ip_nat     TEXT, 
   src_port_nat   INTEGER, 
   dst_ip_nat     TEXT, 
   dst_port_nat   INTEGER, 
   src_intf       TEXT, 
   dst_intf       TEXT, 
   rc             INTEGER, 
   pckt_len       INTEGER, 
   ttl            INTEGER, 
   pr_info        TEXT, 
   proxy_act      TEXT, 
   alarm_name     TEXT, 
   alarm_type     TEXT, 
   alarm_id       TEXT, 
   info_1         TEXT, 
   info_2         TEXT, 
   info_3         TEXT, 
   info_4         TEXT, 
   info_5         TEXT, 
   info_6         TEXT, 
   info_7         TEXT, 
   info_8         TEXT, 
   info_9         TEXT, 
   info_10        TEXT, 
   info_11        TEXT, 
   info_12        TEXT, 
   log_type       TEXT, 
   msg            TEXT, 
   bucket         TEXT, 
   update_time    TIMESTAMP 
);
CREATE INDEX cluster_traffic_utm_idx ON cluster_traffic(update_time);
CREATE INDEX cluster_traffic_ut_idx ON cluster_traffic(sn, "cluster", update_time);
CREATE INDEX cluster_traffic_sip_idx ON cluster_traffic(src_ip);
CREATE INDEX cluster_traffic_sid_idx ON cluster_traffic(sid);

CREATE SEQUENCE cluster_status_sid_seq MINVALUE 1 MAXVALUE 2147483647 CYCLE;
CREATE TABLE cluster_status(
  sid         INTEGER DEFAULT NEXTVAL('cluster_status_sid_seq'), 
  cluster     TEXT DEFAULT '', 
  sn          TEXT NOT NULL, 
  tag_id      INTEGER, 
  raw_id      BIGINT DEFAULT 0, 
  pri         SMALLINT, 
  proc_id     TEXT, 
  msg_id      TEXT, 
  log_type    TEXT, 
  msg         TEXT, 
  update_time TIMESTAMP 
);
CREATE INDEX cluster_status_utm_idx ON cluster_status(update_time);
CREATE INDEX cluster_status_ut_idx ON cluster_status(sn, "cluster", update_time);
CREATE INDEX cluster_status_tid_idx ON cluster_status(tag_id) WHERE tag_id != 1002;

CREATE SEQUENCE cluster_pe_sid_seq MINVALUE 1 MAXVALUE 2147483647 CYCLE;
CREATE TABLE cluster_pe(
  sid         INTEGER DEFAULT NEXTVAL('cluster_pe_sid_seq'), 
  cluster     TEXT DEFAULT '', 
  sn          TEXT NOT NULL, 
  raw_id      BIGINT DEFAULT 0, 
  device      TEXT, 
  unix_time   TEXT, 
  in_octets   BIGINT, 
  out_octets  BIGINT, 
  log_type    TEXT, 
  update_time TIMESTAMP 
);
CREATE INDEX cluster_pe_utm_idx ON cluster_pe(update_time);
CREATE INDEX cluster_pe_ut_idx ON cluster_pe(sn, "cluster", update_time);

CREATE SEQUENCE cluster_wap_info_sid_seq START WITH 1;
CREATE TABLE cluster_wap_info(
  sid           BIGINT NOT NULL DEFAULT nextval('cluster_wap_info_sid_seq') PRIMARY KEY,
  appliance_id  INTEGER NOT NULL,       -- non-cluster appliance id from g_deviceinfo or cluster id from g_clusterinfo
  wap_sn        TEXT NOT NULL,
  wap_name      TEXT,
  status        INTEGER DEFAULT 0,      -- 0: offline, 1: online
  uptime        INTEGER DEFAULT 0,      -- uptime in seconds
  version       TEXT,
  log_type      TEXT,
  update_time   TIMESTAMP NOT NULL
);
CREATE INDEX cluster_wap_info_ut_idx ON cluster_wap_info(update_time);
CREATE INDEX cluster_wap_info_appliance_id_idx ON cluster_wap_info(appliance_id);
CREATE INDEX cluster_wap_info_sn_idx ON cluster_wap_info(wap_sn);

CREATE SEQUENCE cluster_pe_wap_sid_seq START WITH 1;
CREATE TABLE cluster_pe_wap(
  sid           BIGINT NOT NULL DEFAULT nextval('cluster_pe_wap_sid_seq') PRIMARY KEY,
  appliance_id  INTEGER NOT NULL,       -- non-cluster appliance id from g_deviceinfo or cluster id from g_clusterinfo 
  cluster       TEXT NOT NULL,
  sn            TEXT NOT NULL,
  wap_sn        TEXT NOT NULL,
  ssid          TEXT,
  radio         TEXT,
  band          TEXT,
  rcvd_bytes    BIGINT,
  sent_bytes    BIGINT,
  client_count  BIGINT,
  log_type      TEXT,
  update_time   TIMESTAMP NOT NULL
);
CREATE INDEX cluster_pe_wap_appliance_id_idx ON cluster_pe_wap(appliance_id);
CREATE INDEX cluster_pe_wap_wap_sn_idx ON cluster_pe_wap(wap_sn);
CREATE INDEX cluster_pe_wap_ut_idx ON cluster_pe_wap(update_time);
CREATE INDEX cluster_pe_wap_ssid_idx ON cluster_pe_wap(ssid);

CREATE SEQUENCE cluster_srv_sid_seq MINVALUE 1 MAXVALUE 2147483647 CYCLE;
CREATE TABLE cluster_srv(
   sid       INTEGER DEFAULT NEXTVAL('cluster_srv_sid_seq'),
   cluster   TEXT DEFAULT '', 
   sn        TEXT NOT NULL, 
   tag_id    INTEGER, 
   raw_id    BIGINT DEFAULT 0, 
   pri       SMALLINT, 
   conn_id   INTEGER, 
   req_id    INTEGER, 
   mc        INTEGER, 
   rc        INTEGER, 
   app       TEXT, 
   proc_id   TEXT, 
   msg       TEXT, 
   info_1    TEXT, 
   info_2    TEXT, 
   info_3    TEXT, 
   info_4    TEXT, 
  update_time TIMESTAMP 
);
CREATE INDEX cluster_srv_utm_idx ON cluster_srv(update_time);
CREATE INDEX cluster_srv_ut_idx ON cluster_srv(sn, "cluster", update_time);
CREATE INDEX cluster_srv_tid_idx ON cluster_srv(tag_id) WHERE tag_id in (2002, 2005, 2012, 2015);

CREATE TABLE cluster_pe_security(
   sid                     BIGSERIAL PRIMARY KEY,
   cluster                 TEXT,
   sn                      TEXT,
   device                  TEXT DEFAULT 'security',
   raw_id                  BIGINT,
   unix_time               TEXT,
   dlp_allow               INTEGER,
   dlp_drop                INTEGER,
   dlp_quarantine          INTEGER,
   dlp_scanned             INTEGER,
   gav_drop                INTEGER,
   gav_notscanned          INTEGER,
   gav_scanned             INTEGER,
   ips_scanned             INTEGER,
   ips_detected            INTEGER,
   ips_prevented           INTEGER,
   red_bad                 INTEGER,
   red_good                INTEGER,
   red_inconclusive        INTEGER,
   spam_scanned            INTEGER,
   spam_drop               INTEGER,
   spam_tagged             INTEGER,
   spam_quarantined        INTEGER,
   log_type                TEXT,
   update_time             TIMESTAMP,
   apt_scanned             INTEGER,
   apt_prevented           INTEGER,
   apt_notified            INTEGER,
   apt_quarantined         INTEGER,
   apt_zeroday_detected    INTEGER,
   botnet_src_tested       INTEGER,
   botnet_src_dropped      INTEGER,
   botnet_dst_tested       INTEGER,
   botnet_dst_dropped      INTEGER,
   wb_requests             INTEGER,
   wb_allowed              INTEGER,
   wb_denied               INTEGER
);
CREATE INDEX cluster_pe_security_sn_idx ON cluster_pe_security(sn);
CREATE INDEX cluster_pe_security_cl_idx ON cluster_pe_security("cluster");
CREATE INDEX cluster_pe_security_ut_idx ON cluster_pe_security(update_time);

CREATE TABLE cluster_pe_mobile(
   sid                    BIGSERIAL PRIMARY KEY,
   cluster_id             INTEGER DEFAULT 0,
   appliance_id           INTEGER NOT NULL,
   cluster                TEXT NOT NULL,
   sn                     TEXT NOT NULL,
   device_index           BIGINT NOT NULL,
   log_type               TEXT,
   unix_time              TEXT,
   uuid                   TEXT,
   hostname               TEXT,
   ip                     TEXT,
   mac                    TEXT,
   ifname                 TEXT,
   type                   TEXT,
   os_version             TEXT,
   vpn	                  TEXT,
   state                  TEXT,
   source                 TEXT,
   compliance_status      INTEGER,
   compliance_setting     TEXT,
   device_user            TEXT, 
   daa_version            TEXT,
   daa_action             TEXT,
   daa_time               TIMESTAMP,
   update_time            TIMESTAMP NOT NULL
);
CREATE INDEX cluster_pe_mobile_device_index_idx ON cluster_pe_mobile(device_index);
CREATE INDEX cluster_pe_mobile_cluster_id_idx ON cluster_pe_mobile(cluster_id);
CREATE INDEX cluster_pe_mobile_appliance_id_idx ON cluster_pe_mobile(appliance_id);
CREATE INDEX cluster_pe_mobile_hostname_idx ON cluster_pe_mobile(hostname);
CREATE INDEX cluster_pe_mobile_device_user_idx ON cluster_pe_mobile(device_user);
CREATE INDEX cluster_pe_mobile_ut_idx ON cluster_pe_mobile(update_time);

CREATE SEQUENCE cluster_audit_sid_seq MINVALUE 1 MAXVALUE 2147483647 CYCLE;
CREATE SEQUENCE cluster_audit_import_seq START WITH 3;
CREATE TABLE cluster_audit(
   sid                INTEGER NOT NULL DEFAULT NEXTVAL('cluster_audit_sid_seq'), 
   cluster            TEXT DEFAULT '', 
   sn                 TEXT NOT NULL, 
   total_count_period INTEGER,
   lt_tr_count_period INTEGER,
   lt_db_count_period INTEGER,
   lt_ev_count_period INTEGER,
   lt_al_count_period INTEGER,
   lt_pe_count_period INTEGER,
   lt_se_count_period INTEGER DEFAULT 0,
   lt_pw_count_period INTEGER DEFAULT 0,
   lt_pm_count_period INTEGER DEFAULT 0,
   updated            INTEGER DEFAULT 0,
   update_time        TIMESTAMP,
   segment_date       DATE,
   segment_id         TEXT
);
ALTER TABLE cluster_audit ADD CONSTRAINT cluster_audit_pkey PRIMARY KEY (sid, sn);
CREATE INDEX cluster_audit_sn_idx ON cluster_audit (sn);
CREATE INDEX cluster_audit_ut_idx ON cluster_audit (update_time);

CREATE TABLE cluster_diag_audit(
   sid                INTEGER NOT NULL DEFAULT NEXTVAL('cluster_audit_sid_seq'), 
   cluster            TEXT DEFAULT '', 
   sn                 TEXT NOT NULL, 
   total_count_period INTEGER,
   lt_tr_count_period INTEGER,
   lt_db_count_period INTEGER,
   lt_ev_count_period INTEGER,
   lt_al_count_period INTEGER,
   lt_pe_count_period INTEGER,
   lt_se_count_period INTEGER DEFAULT 0,
   lt_pw_count_period INTEGER DEFAULT 0,
   lt_pm_count_period INTEGER DEFAULT 0,
   updated            INTEGER DEFAULT 0,
   update_time        TIMESTAMP,
   segment_date       DATE,
   segment_id         TEXT
);
ALTER TABLE cluster_diag_audit ADD CONSTRAINT cluster_diag_audit_pkey PRIMARY KEY (sid, sn);
CREATE INDEX cluster_diag_audit_sn_idx ON cluster_diag_audit (sn);
CREATE INDEX cluster_diag_audit_ut_idx ON cluster_diag_audit (update_time);
ALTER TABLE cluster_diag_audit INHERIT cluster_audit;

CREATE TABLE summary_traffic(
   sid            BIGSERIAL PRIMARY KEY,
   cluster        TEXT, 
   sn             TEXT NOT NULL, 
   tag_id         INTEGER, 
   src_ip         TEXT, 
   dst_ip         TEXT, 
   dst_port       INTEGER, 
   protocol       TEXT,
   src_intf       TEXT,
   dst_intf       TEXT,
   disp           SMALLINT,
   rcvd_bytes     BIGINT,
   sent_bytes     BIGINT,
   duration       INTEGER,
   count          INTEGER,
   start_time     TIMESTAMP,
   policy_id      TEXT,
   src_user       TEXT,
   dst_domain     TEXT,
   app_id         TEXT,
   signature_id   TEXT,
   deny_type_id   TEXT,
   reputation     TEXT,
   wb_cat_id      TEXT,
   alarm_name     TEXT,
   virus          TEXT,
   sender         TEXT,
   recipients     TEXT,
   host           TEXT,
   dlp_rule_id    TEXT,
   spam_type      TEXT,
   spam_action    TEXT,
   apt_id         TEXT,
   content_name   TEXT,
   quota_limit    TEXT,
   device_index   BIGINT
);
CREATE INDEX summary_traffic_bwsn_idx ON summary_traffic(start_time, tag_id, sn) where tag_id = 1002;
CREATE INDEX summary_traffic_bwcl_idx ON summary_traffic(start_time, tag_id, cluster) where cluster is not null and tag_id = 1002;
CREATE INDEX summary_traffic_dip_idx ON summary_traffic(dst_ip);
CREATE INDEX summary_traffic_sip_idx ON summary_traffic(src_ip);
CREATE INDEX summary_traffic_su_idx ON summary_traffic(src_user) WHERE src_user IS NOT NULL;
CREATE INDEX summary_traffic_spam_idx ON summary_traffic(spam_type) WHERE spam_type IS NOT NULL;
CREATE INDEX summary_traffic_content_name_idx ON summary_traffic(content_name) WHERE content_name IS NOT NULL;
CREATE INDEX summary_traffic_quota_limit_idx ON summary_traffic(quota_limit) WHERE quota_limit IS NOT NULL;
-- Top Clients, Top Destinations, Top Domains, Top Protocols
CREATE INDEX summary_traffic_start_time_sn_tag_id_idx ON summary_traffic(start_time, sn, tag_id) WHERE disp = 1 AND tag_id != 1006;
CREATE INDEX summary_traffic_start_time_cluster_tag_id_idx ON summary_traffic(start_time, cluster, tag_id) WHERE cluster IS NOT NULL AND disp = 1 AND tag_id != 1006;
-- Top URL Categories, Top Blocked URL Categories
CREATE INDEX summary_traffic_start_time_sn_wb_cat_id_idx ON summary_traffic(start_time, sn, wb_cat_id) WHERE wb_cat_id IS NOT NULL;
CREATE INDEX summary_traffic_start_time_cluster_wb_cat_id_idx ON summary_traffic(start_time, cluster, wb_cat_id) WHERE cluster IS NOT NULL AND wb_cat_id IS NOT NULL;
-- Top Applications, Top Application Categories, Top Blocked Applications, Top Blocked Application Categories
CREATE INDEX summary_traffic_start_time_sn_app_id_idx ON summary_traffic(start_time, sn, app_id) WHERE app_id IS NOT NULL;
CREATE INDEX summary_traffic_start_time_cluster_app_id_idx ON summary_traffic(start_time, cluster, app_id) WHERE cluster IS NOT NULL AND app_id IS NOT NULL;
-- Top Blocked Clients, Top Blocked Destinations, Top Blocked Protocols
CREATE INDEX summary_traffic_start_time_sn_deny_idx ON summary_traffic(start_time, sn) WHERE disp != 1;
CREATE INDEX summary_traffic_start_time_cluster_deny_idx ON summary_traffic(start_time, cluster) WHERE cluster IS NOT NULL AND disp != 1;
-- Top Blocked Botnet Sites
CREATE INDEX summary_traffic_start_time_sn_botnet_idx ON summary_traffic(start_time, sn) WHERE tag_id IN (8011, 8012);
CREATE INDEX summary_traffic_start_time_cluster_botnet_idx ON summary_traffic(start_time, cluster) WHERE cluster IS NOT NULL AND tag_id IN (8011, 8012);
-- Top Blocked Malware
CREATE INDEX summary_traffic_start_time_sn_virus_idx ON summary_traffic(start_time, sn) WHERE virus IS NOT NULL;
CREATE INDEX summary_traffic_start_time_cluster_virus_idx ON summary_traffic(start_time, cluster) WHERE cluster IS NOT NULL AND virus IS NOT NULL;
-- Top Blocked Attacks
CREATE INDEX summary_traffic_start_time_sn_signature_id_idx ON summary_traffic(start_time, sn, signature_id) WHERE signature_id IS NOT NULL;
CREATE INDEX summary_traffic_start_time_cluster_signature_id_idx ON summary_traffic(start_time, cluster, signature_id) WHERE cluster IS NOT NULL AND signature_id IS NOT NULL;
-- Top Zero-Day Malware(APT), Top Blocked Advanced Malware(APT)
CREATE INDEX summary_traffic_start_time_sn_apt_id_idx ON summary_traffic(start_time, sn, apt_id) WHERE apt_id IS NOT NULL;
CREATE INDEX summary_traffic_start_time_cluster_apt_id_idx ON summary_traffic(start_time, cluster, apt_id) WHERE cluster IS NOT NULL AND apt_id IS NOT NULL;
-- Data Loss Violations (DLP)
CREATE INDEX summary_traffic_start_time_sn_dlp_rule_id_idx ON summary_traffic(start_time, sn, dlp_rule_id) WHERE dlp_rule_id IS NOT NULL;
CREATE INDEX summary_traffic_start_time_cluster_dlp_rule_id_idx ON summary_traffic(start_time, cluster, dlp_rule_id) WHERE cluster IS NOT NULL AND dlp_rule_id IS NOT NULL;

CREATE TABLE summary_audit(
   sid                     BIGSERIAL PRIMARY KEY,
   sn                      TEXT NOT NULL,
   cluster                 TEXT,
   begin_period            TIMESTAMP NOT NULL,
   rt_count_total          INTEGER,
   rt_count_pf             INTEGER,
   rt_count_px             INTEGER,
   rt_count_wb             INTEGER,
   rt_count_wb_allow       INTEGER,
   rt_count_http           INTEGER,
   rt_count_https          INTEGER,
   rt_count_smtp           INTEGER,
   rt_count_pop3           INTEGER,
   rt_count_ftp            INTEGER,
   rt_count_ips            INTEGER,
   rt_count_av             INTEGER,
   rt_count_app            INTEGER,
   rt_count_app_block      INTEGER,
   rt_count_red            INTEGER,
   rt_count_alarm          INTEGER,
   rt_count_deny           INTEGER,
   rt_count_pe             INTEGER,
   rt_count_dlp            INTEGER,
   rt_count_spam           INTEGER,
   rt_count_apt            INTEGER,
   rt_count_audit          INTEGER,
   rt_count_auth           INTEGER,
   rt_count_wids           INTEGER,
   rt_count_dhcp           INTEGER,
   rt_count_quota          INTEGER,
   rt_count_wapev          INTEGER,
   rt_count_waprgev        INTEGER,
   rt_count_wap_info       INTEGER,
   rt_count_wap_pe         INTEGER,
   rt_count_mobile_pe      INTEGER,
   rt_count_src_botnet     INTEGER,
   rt_count_dst_botnet     INTEGER,
   rt_count_security       INTEGER,
   rt_count_firewall       INTEGER,
   rt_count_default_threat INTEGER,
   rt_count_imap           INTEGER,
   segment_date            DATE,
   segment_id              TEXT
);
CREATE INDEX summary_audit_sn_idx ON summary_audit (sn);
CREATE INDEX summary_audit_cl_idx ON summary_audit ("cluster");
CREATE INDEX summary_audit_bp_idx ON summary_audit (begin_period);

-- used when sending reports to connectwise
CREATE TABLE connectwise_server_config(
   server_id               SERIAL PRIMARY KEY,
   enabled                 INTEGER,
   server_name             TEXT,
   timezone                TEXT,
   server_creds            TEXT,
   company_id              TEXT,
   ca_certificate          TEXT,
   api_keys                TEXT DEFAULT ':'
);

CREATE TABLE connectwise_company_config(
   company_num             SERIAL PRIMARY KEY, 
   company_name            TEXT,
   solution_name           TEXT
);

-- used when sending reports to an ftp server
CREATE TABLE ftp_server_config(
   server_id             SERIAL PRIMARY KEY,
   server_name           TEXT NOT NULL,
   upload_location       TEXT DEFAULT '',
   server_creds          TEXT,
   CONSTRAINT fc UNIQUE(server_name, upload_location)
);

CREATE SEQUENCE scheduled_tasks_sid_seq MINVALUE 1 MAXVALUE 2147483647 CYCLE;

CREATE TABLE scheduled_tasks(
   task_id                INTEGER DEFAULT NEXTVAL('scheduled_tasks_sid_seq') PRIMARY KEY,
   task_type              TEXT NOT NULL,
   task_name              TEXT UNIQUE,
   description            TEXT DEFAULT '',
   run_interval           INTERVAL DEFAULT '0 days',
   notify_admin           BOOLEAN NOT NULL DEFAULT FALSE,
   task_status            INTEGER NOT NULL DEFAULT 0,
   task_progress          INTEGER NOT NULL DEFAULT 0,
   fail_times             INTEGER NOT NULL DEFAULT 0,
   last_failure           TEXT NOT NULL DEFAULT '',
   next_run_time          TIMESTAMP NOT NULL,
   start_run_time         TIMESTAMP,
   last_update_time       TIMESTAMP,
   end_run_time           TIMESTAMP,
   task_modify_time       TIMESTAMP,
   task_create_time       TIMESTAMP NOT NULL,
   task_creator           TEXT
) WITH(FILLFACTOR = 50);
CREATE INDEX scheduled_tasks_tn_idx ON scheduled_tasks(task_name);
CREATE INDEX scheduled_tasks_nrt_idx ON scheduled_tasks(next_run_time);

CREATE TABLE report_tasks_detail(
   task_id                INTEGER REFERENCES scheduled_tasks(task_id) ON DELETE CASCADE,
   task_type              TEXT DEFAULT 'report' CHECK(task_type = 'report'),
   appliances             INTEGER[] NOT NULL,
   report_types           INTEGER[] NOT NULL,
   data_begin_time        TIMESTAMP,
   data_end_time          TIMESTAMP,
   aggregate              TEXT NOT NULL,
   tzname                 TEXT NOT NULL,
   language               TEXT NOT NULL DEFAULT 'en_US',
   customization_name     TEXT,
   report_types_remaining INTEGER[],
   report_types_success   INTEGER[],
   file_target            INTEGER REFERENCES ftp_server_config(server_id),
   file_location          TEXT,
   connectwise_target     INTEGER REFERENCES connectwise_company_config(company_num),
   email_targets          TEXT[],
   email_subject          TEXT,
   email_body             TEXT
) WITH(FILLFACTOR = 50);
CREATE INDEX report_tasks_detail_tid_idx ON report_tasks_detail(task_id);

CREATE TABLE segment_tasks_detail(
   task_id                INTEGER REFERENCES scheduled_tasks(task_id) ON DELETE CASCADE,
   task_type              TEXT NOT NULL,
   data_begin_time        TIMESTAMP NOT NULL,
   data_end_time          TIMESTAMP,
   segments               TEXT[],
   segments_success       TEXT[],
   segments_remaining     TEXT[]
) WITH(FILLFACTOR = 50);
CREATE INDEX segment_tasks_detail_tid_idx ON segment_tasks_detail(task_id);

CREATE TABLE report_customization(
   customization_name      TEXT PRIMARY KEY,
   header                  TEXT,
   footer                  TEXT,
   logo                    BYTEA,
   is_default              BOOLEAN,
   paper_size              INTEGER NOT NULL DEFAULT 0
);

-- maps application ids in the summary_traffic table to application category and name for the application control reports
CREATE TABLE g_appinfo (
   app_id       TEXT PRIMARY KEY,
   app_name     TEXT,
   app_cat_name TEXT);
CREATE INDEX g_appinfo_an_idx ON g_appinfo(app_name);
CREATE INDEX g_appinfo_acn_idx ON g_appinfo(app_cat_name);

-- maps IPS signature names and categories to an id in the summary_traffic_table for IPS reports
CREATE TABLE g_ipsinfo (
   signature_id   TEXT PRIMARY KEY,
   signature_name TEXT,
   signature_cat  TEXT,
   severity       TEXT);
CREATE INDEX g_ipsinfo_sn_idx ON g_ipsinfo(signature_name);
CREATE INDEX g_ipsinfo_sc_idx ON g_ipsinfo(signature_cat);

-- maps a DLP rule to an id in the summary_traffic table
CREATE TABLE g_dlpinfo (
   rule_id        TEXT PRIMARY KEY,
   rule_name      TEXT);
CREATE INDEX g_dlpinfo_rn_idx on g_dlpinfo(rule_name);
CREATE SEQUENCE g_dlpinfo_id_seq MINVALUE 1;

-- maps a deny type (i.e. unhandled external packet, tcp syn flooding) to an id in the summary_traffic table
CREATE TABLE g_denyinfo (
   type_id        TEXT PRIMARY KEY,
   type_name      TEXT,
   type_cat       TEXT);
CREATE INDEX g_denyinfo_tn_idx on g_denyinfo(type_name);
CREATE SEQUENCE g_denyinfo_id_seq MINVALUE 1;

CREATE TABLE g_aptinfo (
   apt_id         TEXT PRIMARY KEY,
   threat_uuid    TEXT NOT NULL,
   threat_md5     TEXT NOT NULL,
   threat_level   TEXT NOT NULL);
CREATE INDEX g_aptinfo_threat_uuid_idx ON g_aptinfo(threat_uuid);
CREATE INDEX g_aptinfo_threat_md5_idx ON g_aptinfo(threat_md5);
CREATE INDEX g_aptinfo_threat_level_idx ON g_aptinfo(threat_level);
CREATE SEQUENCE g_aptinfo_id_seq MINVALUE 1;

-- contains info about APT threats
CREATE TABLE g_apt_classification (
   apt_id           TEXT NOT NULL,
   threat_class     TEXT,
   threat_name      TEXT,
   threat_score     TEXT,
   threat_mime_type TEXT);
ALTER TABLE g_apt_classification ADD CONSTRAINT g_apt_classification_apt_id_fkey FOREIGN KEY (apt_id) REFERENCES g_aptinfo(apt_id);
CREATE INDEX g_apt_classification_apt_id_idx ON g_apt_classification(apt_id);
CREATE INDEX g_apt_classification_threat_class_idx ON g_apt_classification(threat_class);
CREATE INDEX g_apt_classification_threat_name_idx ON g_apt_classification(threat_name);
CREATE INDEX g_apt_classification_threat_score_idx ON g_apt_classification(threat_score);
CREATE INDEX g_apt_classification_threat_mime_type_idx ON g_apt_classification(threat_mime_type);

CREATE TABLE g_apt_activity_info (
   activity_id    TEXT PRIMARY KEY,
   activity       TEXT);
CREATE sequence g_apt_activity_info_id_seq minvalue 1;

-- maps APT threats to a list of malicious activities for the APT reports
CREATE TABLE g_apt_activities (
   apt_id         TEXT NOT NULL,
   activity_id    TEXT NOT NULL);
ALTER TABLE g_apt_activities ADD CONSTRAINT g_apt_activities_apt_id_fkey FOREIGN KEY (apt_id) REFERENCES g_aptinfo(apt_id);
ALTER TABLE g_apt_activities ADD CONSTRAINT g_apt_activities_activity_id_fkey FOREIGN KEY (activity_id) REFERENCES g_apt_activity_info(activity_id);

CREATE TABLE apt_submissions(
   sn             TEXT NOT NULL,
   cluster        TEXT,
   apt_id         TEXT NOT NULL,
   start_time     TIMESTAMP, 
   end_time       TIMESTAMP);
CREATE INDEX apt_submissions_sn_idx ON apt_submissions(sn);
CREATE INDEX apt_submissions_cluster_idx ON apt_submissions(cluster);
CREATE INDEX apt_submissions_apt_id_idx ON apt_submissions(apt_id);
CREATE INDEX apt_submissions_start_time_idx ON apt_submissions(start_time);
CREATE INDEX apt_submissions_end_time_idx ON apt_submissions(end_time);

CREATE TABLE apt_submissions_0(
   sn             TEXT NOT NULL,
   cluster        TEXT,
   apt_id         TEXT NOT NULL,
   start_time     TIMESTAMP, 
   end_time       TIMESTAMP);
CREATE INDEX apt_submissions_0_sn_idx ON apt_submissions_0(sn);
CREATE INDEX apt_submissions_0_cluster_idx ON apt_submissions_0(cluster);
CREATE INDEX apt_submissions_0_apt_id_idx ON apt_submissions_0(apt_id);
CREATE INDEX apt_submissions_0_start_time_idx ON apt_submissions_0(start_time);
CREATE INDEX apt_submissions_0_end_time_idx ON apt_submissions_0(end_time);
ALTER TABLE apt_submissions_0 INHERIT apt_submissions;

-- maps WebBlocker category name to an id in the summary_traffic table for reports
CREATE TABLE g_wbinfo (
   cat_id         TEXT PRIMARY KEY,
   cat_name       TEXT);
CREATE INDEX g_wbinfo_cn_idx on g_wbinfo(cat_name);
CREATE SEQUENCE g_wbinfo_id_seq MINVALUE 100000;

-- maps a firebox policy name to an id in the summary_traffic table for reports
CREATE TABLE g_policyinfo (
   policy_id      TEXT PRIMARY KEY,
   policy_name    TEXT);
CREATE INDEX g_policyinfo_pn_idx on g_policyinfo(policy_name);
CREATE SEQUENCE g_policyinfo_id_seq MINVALUE 1;

-- used by reports to map ip addresses to host names
CREATE TABLE g_hostmap(
   appliance_id     INTEGER,
   ip               INET,
   fqdn             TEXT,
   resolve_time     TIMESTAMP, 
   expire_time      TIMESTAMP,
   UNIQUE(appliance_id, ip));

CREATE INDEX g_hostmap_appliance_id_idx ON g_hostmap(appliance_id);
CREATE INDEX g_hostmap_ip_idx ON g_hostmap(ip);
CREATE INDEX g_hostmap_resolve_time_idx ON g_hostmap(resolve_time);
CREATE INDEX g_hostmap_expire_time_idx ON g_hostmap(expire_time);

CREATE TABLE g_hostmap_static(
   appliance_id     INTEGER,
   ip               INET,
   fqdn             TEXT,
   resolve_time     TIMESTAMP, 
   expire_time      TIMESTAMP,
   UNIQUE(appliance_id, ip));

CREATE INDEX g_hostmap_static_appliance_id_idx ON g_hostmap(appliance_id);
CREATE INDEX g_hostmap_static_ip_idx ON g_hostmap(ip);
CREATE INDEX g_hostmap_static_resolve_time_idx ON g_hostmap(resolve_time);
CREATE INDEX g_hostmap_static_expire_time_idx ON g_hostmap(expire_time);
ALTER TABLE g_hostmap_static INHERIT g_hostmap;

CREATE TABLE g_hostmap_work(
   ip               INET,
   input_time       TIMESTAMP,
   input_table      TEXT);

CREATE INDEX g_hostmap_work_idx ON g_hostmap_work(ip, input_time, input_table);

CREATE TABLE g_HostMapTable(
   ip               INET,
   fqdn             VARCHAR(255),
   expire_time      TIMESTAMP,
   update_time      TIMESTAMP);

CREATE INDEX g_HostMapIndex_ip ON g_HostMapTable(ip);
CREATE INDEX g_HostMapIndex_expire_time ON g_HostMapTable(expire_time);
CREATE INDEX g_HostMapIndex_update_time ON g_HostMapTable(update_time);

-- should be deprecated
CREATE TABLE g_FireboxDeletionTable(
   sn               TEXT NOT NULL,
   cluster          TEXT DEFAULT '',
   type             SMALLINT NOT NULL,
   PRIMARY KEY (sn, cluster)
);

-- map a tag in an XML log, such as ProxyHTTPReq to an id that is stored in the log data tables
CREATE TABLE g_MessageTagTable(
   tag_id           INTEGER PRIMARY KEY,
   tag_text         TEXT    
);
-- g_messagetagtable is populated after the schema is loaded

-- should be deprecated
CREATE TABLE g_sn_TimeTable (
    sn                 TEXT PRIMARY KEY,
    first_record_time  TIMESTAMP,
    last_record_time   TIMESTAMP
);

-- active logging clients with the log collector id
CREATE TABLE g_sn_CollectorTable (
    id                 SERIAL PRIMARY KEY,
    sn                 TEXT,
    collector_id       TEXT NOT NULL,
    update_time        TIMESTAMP DEFAULT date_trunc('second', clock_timestamp() AT TIME ZONE 'UTC')
);
CREATE INDEX g_sn_collectortable_sn_idx ON g_sn_collectortable (sn);
CREATE INDEX g_sn_collectortable_cid_idx ON g_sn_collectortable (collector_id);
CREATE INDEX g_sn_collectortable_ut_idx ON g_sn_collectortable (update_time);

-- contains list of segments that contain log data (YYYYMMDD_<id>)
CREATE TABLE g_partitiontable (
    segment_id         INTEGER NOT NULL,
    create_time        TIMESTAMP NOT NULL DEFAULT date_trunc('second', CURRENT_TIMESTAMP AT TIME ZONE 'UTC'),
    data_time          TIMESTAMP NOT NULL DEFAULT date_trunc('second', CURRENT_TIMESTAMP AT TIME ZONE 'UTC'),
    backup_status      INTEGER NOT NULL DEFAULT -1,
    backup_status_time TIMESTAMP NOT NULL DEFAULT date_trunc('second', CURRENT_TIMESTAMP AT TIME ZONE 'UTC')
) WITH (FILLFACTOR = 50);
CREATE INDEX g_partitiontable_ct_idx ON g_partitiontable(create_time);
CREATE INDEX g_partitiontable_dt_idx ON g_partitiontable(data_time);

-- contains the log collectors that are using the database
CREATE TABLE g_log_collectors (
    id                 SERIAL PRIMARY KEY,
    collector_id       TEXT NOT NULL,
    ip                 TEXT NOT NULL DEFAULT host(inet_client_addr())::text,
    update_time        TIMESTAMP DEFAULT date_trunc('second', clock_timestamp() AT TIME ZONE 'UTC')
);
CREATE INDEX g_log_collectors_cid_idx ON g_log_collectors(collector_id);
CREATE INDEX g_log_collectors_ut_idx ON g_log_collectors(update_time);
